<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>iStorya</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <!-- Local Bootstrap CSS -->
  <link href="libs/bootstrap.min.css" rel="stylesheet">

  <!-- Local Poppins Fonts -->
  <style>
    @font-face {
      font-family: 'Poppins';
      src: url('fonts/Poppins-Regular.woff2') format('woff2'),
           url('fonts/Poppins-Regular.woff') format('woff');
      font-weight: 400;
      font-style: normal;
    }
    @font-face {
      font-family: 'Poppins';
      src: url('fonts/Poppins-SemiBold.woff2') format('woff2'),
           url('fonts/Poppins-SemiBold.woff') format('woff');
      font-weight: 600;
      font-style: normal;
    }
    body {
      font-family: 'Poppins', sans-serif;
      background-color: #0a1e3a;
      color: #ffffff;
      margin: 0;
    }
    .navbar {
      background-color: #06172e;
      padding: 10px 20px;
      display: flex;
      justify-content: space-between;
      align-items: center;
    }
    .navbar-brand {
      font-weight: bold;
      color: #6ec1e4;
    }
    .offline-btn {
      background-color: #2fa4e7;
      color: white;
      border: none;
      border-radius: 5px;
      padding: 8px 15px;
      font-size: 0.9rem;
      cursor: pointer;
      transition: background-color 0.3s;
    }
    .offline-btn:hover {
      background-color: #1c8ad9;
    }
    .hero {
      background: url('exhibit1.jpg') no-repeat center center;
      background-size: cover;
      height: 450px;
      display: flex;
      flex-direction: column;
      justify-content: flex-end;
      padding: 30px;
      position: relative;
    }
    .hero::after {
      content: '';
      position: absolute;
      inset: 0;
      background: linear-gradient(to top, rgba(10, 30, 58, 0.95), rgba(10, 30, 58, 0.3));
    }
    .hero-content {
      position: relative;
      z-index: 1;
    }
    .hero h1 {
      font-weight: bold;
      font-size: 2rem;
      color: #6ec1e4;
    }
    .btn-watch {
      background: #2fa4e7;
      border: none;
      padding: 10px 20px;
      color: white;
      font-weight: bold;
      border-radius: 5px;
      margin-top: 10px;
    }
    .exhibit-cards {
      padding: 20px;
    }
    .exhibit-card {
      background: #112a4d;
      border-radius: 10px;
      overflow: hidden;
      margin-bottom: 20px;
    }
    .exhibit-card img {
      width: 100%;
      height: 150px;
    }
    .loader {
      display: none;
      margin-top: 20px;
    }
    .spinner-border {
      width: 2rem;
      height: 2rem;
    }
    #modeIndicator {
      font-size: 0.8rem;
      color: #aad3e6;
      margin-left: 10px;
    }
    /* Sync overlay */
    #syncOverlay {
      display:none;
      position:fixed;
      top:0; left:0; right:0; bottom:0;
      background:rgba(10,30,58,0.95);
      z-index:9999;
      color:white;
      text-align:center;
      padding-top:20%;
    }
  </style>
</head>
<body>
  <nav class="navbar">
    <span class="navbar-brand">iStorya</span>
    <div>
      <button class="offline-btn" id="modeBtn">Go Offline</button>
      <span id="modeIndicator">(Online Mode)</span>
    </div>
  </nav>

  <section class="hero">
    <div class="hero-content">
      <h1>Welcome to the iStorya Experience</h1>
      <p>Explore and discover stories behind every exhibit.</p>
      <button class="btn-watch" onclick="startRedirect()">Proceed Now</button>
    </div>
    <div class="loader" style="display:none;">
      <div class="spinner-border text-success" role="status">
        <span class="sr-only">Loading...</span>
      </div>
      <p class="mt-2">Preparing Experience...</p>
    </div>
  </section>
   <section class="exhibit-cards container">
    <div class="d-flex flex-row flex-wrap justify-content-between">
      <div class="exhibit-card" style="width: 50%; height: 150px">
        <img src="exhibit2.jpg" alt="Exhibit">
      </div>
      <div class="exhibit-card" style="width: 50%; height: 150px">
        <img src="exhibit3.png" alt="Exhibit">
      </div>
    </div>
  </section>

  <!-- Sync Overlay -->
  <div id="syncOverlay" style="display:none; text-align:center;">
    <h3 id="syncHeader">Preparing for offline experience...</h3>
    <div class="progress mt-3" style="width:60%; margin:20px auto;">
      <div id="syncProgress" class="progress-bar progress-bar-striped progress-bar-animated" 
           role="progressbar" style="width: 0%">0%</div>
    </div>
    <p id="syncStatus">This loading takes a while, please wait...</p>
    <button id="browseBtn" class="btn btn-success mt-4" style="display:none;">
      🚀 Browse the Experience
    </button>
  </div>

  <!-- Dexie JS -->
<script src="libs/dexie.min.js"></script>

<script>
  // --- Step 1: Clear Cache + IndexedDB on Entry ---
  async function clearCachesAndDB() {
    console.log("♻️ Clearing old caches and IndexedDB...");

    // Clear Service Worker cache
    if ('caches' in window) {
      const cacheNames = await caches.keys();
      if (cacheNames.length === 0) {
        console.log("ℹ️ No caches found, nothing to delete.");
      } else {
        for (const name of cacheNames) {
          const deleted = await caches.delete(name);
          console.log(deleted ? `🗑️ Cache "${name}" deleted successfully.` : `⚠️ Failed to delete cache "${name}".`);
        }
      }
    }

    // Clear Dexie DB
    try {
      await Dexie.delete("iStoryaDB");
      console.log("✅ IndexedDB cleared successfully.");
    } catch (err) {
      console.error("❌ Failed to clear IndexedDB:", err);
    }
  }

  // --- Step 2: Continue with your normal setup ---
  // Redirect with loader
  function startRedirect() {
    document.querySelector('.btn-watch').style.display = 'none';
    document.querySelector('.loader').style.display = 'block';
    setTimeout(() => window.location.href = 'setup.php', 1500);
  }

  // IndexedDB Setup
  const db = new Dexie("iStoryaDB");
  db.version(1).stores({
    academic_list: "id",
    assessment_answers: "id",
    assessment_questions: "id",
    class_list: "id",
    department: "id",
    diorama: "id",
    system_settings: "id",
    users: "id"
  });

  function isOnline() { return navigator.onLine; }

  async function cacheMedia(url) {
    try {
      const res = await fetch(url);
      const blob = await res.blob();
      return await new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onloadend = () => resolve(reader.result);
        reader.onerror = reject;
        reader.readAsDataURL(blob);
      });
    } catch (err) {
      console.warn("⚠️ Media fetch failed:", url, err);
      return null;
    }
  }

  const tableDefinitions = {
    academic_list: [],
    assessment_answers: [],
    assessment_questions: [],
    class_list: [],
    department: [],
    diorama: ["image", "audio"],
    system_settings: [],
    users: ["avatar"]
  };

  function renderTable(table, data) {
    console.log(`📋 Rendering ${table}: ${data.length} records`);
  }

  async function syncTable(table, mediaFields = []) {
    try {
      if (isOnline()) {
        const res = await fetch("api/get_" + table + ".php");
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const data = await res.json();

        if (Array.isArray(data) && data.length > 0) {
          const processed = [];
          for (let row of data) {
            let newRow = { ...row };
            for (let field of mediaFields) {
              if (row[field]) {
                const cached = await cacheMedia(row[field]);
                if (cached) newRow[field + "_blob"] = cached;
              }
            }
            processed.push(newRow);
          }
          await db[table].clear();
          await db[table].bulkPut(processed);
          renderTable(table, processed);
        }
        console.log("✅ Synced:", table);
      } else throw new Error("Offline mode");
    } catch (err) {
      console.warn(`⚠️ Using cached data for ${table} due to:`, err.message);
      const cachedData = await db[table].toArray();
      if (cachedData.length > 0) {
        console.log(`📦 Loaded ${cachedData.length} records from cache for ${table}`);
        renderTable(table, cachedData);
      } else console.error(`❌ No cached data found for ${table}`);
    }
  }

  async function syncAllTablesWithProgress() {
    const tables = Object.keys(tableDefinitions);
    const progressBar = document.getElementById("syncProgress");
    const status = document.getElementById("syncStatus");
    const header = document.getElementById("syncHeader");
    const browseBtn = document.getElementById("browseBtn");

    for (let i = 0; i < tables.length; i++) {
      const t = tables[i];
      status.textContent = "Syncing table: " + t + "...";
      await syncTable(t, tableDefinitions[t]);
      let percent = Math.round(((i + 1) / tables.length) * 100);
      progressBar.style.width = percent + "%";
      progressBar.textContent = percent + "%";
    }

    header.textContent = "🎉 Offline experience is ready!";
    status.textContent = "All data synced successfully.";
    browseBtn.style.display = "inline-block";
    browseBtn.addEventListener("click", () => window.location.href = "setup.php");
  }

  async function syncAllTables() {
    for (let t of Object.keys(tableDefinitions)) {
      await syncTable(t, tableDefinitions[t]);
    }
  }

  const modeBtn = document.getElementById('modeBtn');
  const indicator = document.getElementById('modeIndicator');
  let mode = localStorage.getItem("iStoryaMode") || "online";
  updateUI();

  modeBtn.addEventListener("click", async () => {
    if (mode === "online") {
      document.getElementById("syncOverlay").style.display = "block";
      await syncAllTablesWithProgress();
      mode = "offline";
      localStorage.setItem("iStoryaMode", mode);
      updateUI();
    } else {
      mode = "online";
      localStorage.setItem("iStoryaMode", mode);
      updateUI();
      await syncAllTables();
    }
  });

  function updateUI() {
    if (mode === "offline") {
      modeBtn.textContent = "Go Online";
      indicator.textContent = "(Offline Mode)";
      indicator.style.color = "#00ff99";
    } else {
      modeBtn.textContent = "Go Offline";
      indicator.textContent = "(Online Mode)";
      indicator.style.color = "#aad3e6";
    }
  }

  window.addEventListener("load", async () => {
    await clearCachesAndDB(); // 🆕 Clear cache + IndexedDB on every fresh load

    if (!isOnline()) {
      console.log("⚡ Detected offline — forcing offline mode.");
      mode = "offline";
      localStorage.setItem("iStoryaMode", "offline");
      updateUI();
    }
    if (mode === "online" && isOnline()) {
      await syncAllTables();
      console.log("🌐 Online mode sync complete (silent).");
    } else {
      console.log("📦 Using offline cache for tables.");
      for (let t of Object.keys(tableDefinitions)) {
        const cachedData = await db[t].toArray();
        renderTable(t, cachedData);
      }
    }
  });

  window.addEventListener("offline", () => {
    console.log("⚡ Lost internet, switching to offline.");
    mode = "offline";
    localStorage.setItem("iStoryaMode", "offline");
    updateUI();
  });

  window.addEventListener("online", async () => {
    console.log("🌐 Internet restored, back to online mode.");
    mode = "online";
    localStorage.setItem("iStoryaMode", "online");
    updateUI();
    await syncAllTables();
  });

  if ('serviceWorker' in navigator) {
    window.addEventListener('load', () => {
      navigator.serviceWorker.register('sw.js')
        .then(reg => console.log("✅ Service Worker registered:", reg.scope))
        .catch(err => console.error("❌ SW registration failed:", err));
    });
  }
</script>
</body>

</html>
